"""
Resume Bullet Point Generator
Generates 3 UNIQUE, ATS-friendly bullet points based on job title and description
Uses template-based generation with keyword extraction
"""

import re
from collections import Counter

# Action verb categories
ACTION_VERBS = {
    'leadership': ['Led', 'Directed', 'Managed', 'Supervised', 'Coordinated', 'Orchestrated', 'Spearheaded', 'Championed'],
    'achievement': ['Achieved', 'Exceeded', 'Delivered', 'Accomplished', 'Attained', 'Secured', 'Generated', 'Produced'],
    'improvement': ['Improved', 'Enhanced', 'Optimized', 'Streamlined', 'Upgraded', 'Refined', 'Revolutionized', 'Transformed'],
    'creation': ['Developed', 'Created', 'Designed', 'Built', 'Established', 'Launched', 'Implemented', 'Initiated'],
    'analysis': ['Analyzed', 'Evaluated', 'Assessed', 'Researched', 'Investigated', 'Examined', 'Audited', 'Reviewed'],
    'communication': ['Presented', 'Communicated', 'Collaborated', 'Partnered', 'Liaised', 'Consulted', 'Advised', 'Negotiated'],
    'technical': ['Programmed', 'Engineered', 'Configured', 'Automated', 'Integrated', 'Deployed', 'Maintained', 'Troubleshot']
}

def extract_keywords(text):
    """Extract important keywords from job description"""
    # Remove common words
    stop_words = {'the', 'a', 'an', 'and', 'or', 'but', 'in', 'on', 'at', 'to', 'for', 'of', 'with', 'by'}
    words = re.findall(r'\b[a-zA-Z]{3,}\b', text.lower())
    keywords = [w for w in words if w not in stop_words]
    
    # Get most common keywords
    counter = Counter(keywords)
    return [word for word, count in counter.most_common(10)]

def determine_verb_category(designation, description):
    """Determine which action verb category fits best"""
    combined_text = (designation + ' ' + description).lower()
    
    # Leadership roles
    if any(word in combined_text for word in ['manager', 'lead', 'director', 'supervisor', 'head']):
        return 'leadership'
    
    # Technical roles
    if any(word in combined_text for word in ['developer', 'engineer', 'programmer', 'technical', 'software']):
        return 'technical'
    
    # Creative roles
    if any(word in combined_text for word in ['designer', 'creative', 'artist', 'writer', 'content']):
        return 'creation'
    
    # Analytical roles
    if any(word in combined_text for word in ['analyst', 'researcher', 'data', 'scientist']):
        return 'analysis'
    
    # Sales/Communication roles
    if any(word in combined_text for word in ['sales', 'marketing', 'customer', 'client', 'support']):
        return 'communication'
    
    # Default
    return 'achievement'

def generate_bullet_structure(verb, designation, keywords, tone, index):
    """Generate a single unique bullet point"""
    # Different structures for variety
    structures = [
        # Structure 1: Action + Task + Result
        f"{verb} {keywords[index % len(keywords)]} initiatives for {designation.lower()} operations, resulting in improved {keywords[(index+1) % len(keywords)]} and enhanced team performance",
        
        # Structure 2: Action + Metric + Impact
        f"{verb} comprehensive {keywords[index % len(keywords)]} strategies, achieving 30% increase in {keywords[(index+2) % len(keywords)]} efficiency and stakeholder satisfaction",
        
        # Structure 3: Action + Collaboration + Outcome
        f"{verb} cross-functional collaboration on {keywords[index % len(keywords)]} projects, delivering measurable improvements in {keywords[(index+3) % len(keywords)]} quality and workflow optimization"
    ]
    
    # Select structure based on index
    bullet = structures[index % len(structures)]
    
    # Adjust for tone
    if tone == 'detailed':
        # Add more context
        bullet += f", while maintaining focus on {keywords[(index+4) % len(keywords)]} best practices"
    
    return bullet

def generate_resume_bullets(designation, description, tone='concise'):
    """
    Generate 3 UNIQUE resume bullet points
    
    Args:
        designation: Job title
        description: Job responsibilities/description
        tone: 'concise' or 'detailed'
    
    Returns:
        List of 3 unique bullet points
    """
    try:
        # Extract keywords from description
        keywords = extract_keywords(description)
        if len(keywords) < 3:
            keywords = ['responsibilities', 'tasks', 'projects', 'goals', 'deliverables']
        
        # Determine appropriate verb category
        verb_category = determine_verb_category(designation, description)
        verbs = ACTION_VERBS.get(verb_category, ACTION_VERBS['achievement'])
        
        # Generate 3 unique bullets
        bullets = []
        used_verbs = set()
        
        for i in range(3):
            # Select unique verb
            verb = verbs[i % len(verbs)]
            while verb in used_verbs and len(used_verbs) < len(verbs):
                verb = verbs[(i + len(used_verbs)) % len(verbs)]
            used_verbs.add(verb)
            
            # Generate bullet with unique structure
            bullet = generate_bullet_structure(verb, designation, keywords, tone, i)
            bullets.append(bullet)
        
        return bullets
        
    except Exception as e:
        # Fallback bullets if generation fails
        return [
            f"Managed {designation.lower()} responsibilities with focus on quality and efficiency",
            f"Delivered key projects and initiatives aligned with organizational goals",
            f"Collaborated with cross-functional teams to achieve measurable results"
        ]

# For testing
if __name__ == "__main__":
    # Test
    bullets = generate_resume_bullets(
        "Software Engineer",
        "Develop web applications using React and Node.js. Work with databases and APIs. Collaborate with team members.",
        "concise"
    )
    for i, bullet in enumerate(bullets, 1):
        print(f"{i}. {bullet}")
