"""
AI Presentation Generator - Production Version
Generates complete presentations with title, content, speaker notes, and images
Exports to PPTX, PDF, and DOCX formats
Uses free APIs: Unsplash for images, OpenAI-compatible models
"""

import os
import requests
from datetime import datetime
from typing import Dict, List, Any
from io import BytesIO
import json

# For PPTX generation
from pptx import Presentation
from pptx.util import Inches, Pt
from pptx.enum.text import PP_ALIGN
from pptx.dml.color import RGBColor

# For PDF generation
from reportlab.lib.pagesizes import letter, landscape
from reportlab.lib.units import inch
from reportlab.platypus import SimpleDocTemplate, Paragraph, Spacer, Image as RLImage, PageBreak
from reportlab.lib.styles import getSampleStyleSheet, ParagraphStyle
from reportlab.lib.enums import TA_CENTER, TA_LEFT

# For DOCX generation
from docx import Document
from docx.shared import Inches as DocxInches, Pt as DocxPt, RGBColor as DocxRGB
from docx.enum.text import WD_ALIGN_PARAGRAPH

# AI Generation (using free/open models)
import google.generativeai as genai

# Environment configuration
UNSPLASH_ACCESS_KEY = os.getenv('UNSPLASH_ACCESS_KEY', '')  # Optional
PEXELS_API_KEY = os.getenv('PEXELS_API_KEY', '')  # Optional
GEMINI_API_KEY = os.getenv('GEMINI_API_KEY', '')  # Required for AI

class PresentationGenerator:
    """Generate complete presentations with AI content and images"""
    
    def __init__(self):
        self.temp_dir = '/tmp/presentations'
        os.makedirs(self.temp_dir, exist_ok=True)
        
        # Initialize Gemini AI
        if GEMINI_API_KEY:
            genai.configure(api_key=GEMINI_API_KEY)
            self.model = genai.GenerativeModel('gemini-pro')
        else:
            self.model = None
    
    def generate_outline(self, topic: str, slide_count: int) -> List[Dict[str, Any]]:
        """Generate presentation outline with AI"""
        
        if not self.model:
            # Fallback to template-based generation
            return self._generate_template_outline(topic, slide_count)
        
        prompt = f"""Create a detailed outline for a {slide_count}-slide presentation on: {topic}

For each slide, provide:
1. Slide title (clear and engaging)
2. 3-5 bullet points (concise, informative)
3. Speaker notes (2-3 sentences explaining the slide)
4. Image keyword (1-2 words for finding a relevant image)

Format as JSON array:
[
  {{
    "slide_number": 1,
    "title": "Presentation Title",
    "bullets": ["Point 1", "Point 2", "Point 3"],
    "speaker_notes": "Introduction notes...",
    "image_keyword": "keyword"
  }},
  ...
]

Make the first slide a title slide and the last slide a conclusion/thank you slide.
Content should be professional, informative, and well-structured."""

        try:
            response = self.model.generate_content(prompt)
            text = response.text.strip()
            
            # Extract JSON from response
            if '```json' in text:
                text = text.split('```json')[1].split('```')[0].strip()
            elif '```' in text:
                text = text.split('```')[1].split('```')[0].strip()
            
            outline = json.loads(text)
            
            # Validate and ensure correct count
            if len(outline) != slide_count:
                outline = outline[:slide_count] if len(outline) > slide_count else outline
            
            return outline
            
        except Exception as e:
            print(f"AI generation error: {e}")
            return self._generate_template_outline(topic, slide_count)
    
    def _generate_template_outline(self, topic: str, slide_count: int) -> List[Dict[str, Any]]:
        """Fallback template-based outline generation"""
        
        outline = []
        
        # Title slide
        outline.append({
            "slide_number": 1,
            "title": topic,
            "bullets": [
                "A comprehensive overview",
                f"Presented in {slide_count} slides",
                f"Date: {datetime.now().strftime('%B %Y')}"
            ],
            "speaker_notes": f"Welcome to this presentation on {topic}. We'll cover the key aspects in detail.",
            "image_keyword": topic.split()[0]
        })
        
        # Content slides
        for i in range(2, slide_count):
            outline.append({
                "slide_number": i,
                "title": f"{topic} - Part {i-1}",
                "bullets": [
                    f"Key point {j+1} about {topic}" for j in range(4)
                ],
                "speaker_notes": f"This slide covers important aspects of {topic}.",
                "image_keyword": topic.split()[min(i-2, len(topic.split())-1)]
            })
        
        # Conclusion slide
        outline.append({
            "slide_number": slide_count,
            "title": "Conclusion & Thank You",
            "bullets": [
                f"Recap of {topic}",
                "Key takeaways",
                "Questions welcome"
            ],
            "speaker_notes": f"Thank you for your attention. Let's recap the main points about {topic}.",
            "image_keyword": "thank you"
        })
        
        return outline
    
    def fetch_image(self, keyword: str) -> bytes:
        """Fetch image from Unsplash or Pexels"""
        
        # Try Unsplash first
        if UNSPLASH_ACCESS_KEY:
            try:
                url = f"https://api.unsplash.com/photos/random"
                params = {
                    'query': keyword,
                    'orientation': 'landscape',
                    'client_id': UNSPLASH_ACCESS_KEY
                }
                response = requests.get(url, params=params, timeout=10)
                if response.status_code == 200:
                    data = response.json()
                    img_url = data['urls']['regular']
                    img_response = requests.get(img_url, timeout=10)
                    if img_response.status_code == 200:
                        return img_response.content
            except Exception as e:
                print(f"Unsplash error: {e}")
        
        # Try Pexels
        if PEXELS_API_KEY:
            try:
                url = "https://api.pexels.com/v1/search"
                headers = {'Authorization': PEXELS_API_KEY}
                params = {'query': keyword, 'per_page': 1, 'orientation': 'landscape'}
                response = requests.get(url, headers=headers, params=params, timeout=10)
                if response.status_code == 200:
                    data = response.json()
                    if data['photos']:
                        img_url = data['photos'][0]['src']['large']
                        img_response = requests.get(img_url, timeout=10)
                        if img_response.status_code == 200:
                            return img_response.content
            except Exception as e:
                print(f"Pexels error: {e}")
        
        # Return placeholder
        return self._create_placeholder_image(keyword)
    
    def _create_placeholder_image(self, keyword: str) -> bytes:
        """Create a simple placeholder image"""
        from PIL import Image, ImageDraw, ImageFont
        
        img = Image.new('RGB', (1280, 720), color=(100, 100, 150))
        draw = ImageDraw.Draw(img)
        
        try:
            font = ImageFont.truetype("/usr/share/fonts/truetype/dejavu/DejaVuSans-Bold.ttf", 48)
        except:
            font = ImageFont.load_default()
        
        text = f"Image: {keyword}"
        bbox = draw.textbbox((0, 0), text, font=font)
        text_width = bbox[2] - bbox[0]
        text_height = bbox[3] - bbox[1]
        
        position = ((1280 - text_width) // 2, (720 - text_height) // 2)
        draw.text(position, text, fill=(255, 255, 255), font=font)
        
        buffer = BytesIO()
        img.save(buffer, format='PNG')
        return buffer.getvalue()
    
    def generate_pptx(self, outline: List[Dict], topic: str, add_watermark: bool = False) -> str:
        """Generate PowerPoint presentation"""
        
        prs = Presentation()
        prs.slide_width = Inches(10)
        prs.slide_height = Inches(7.5)
        
        for slide_data in outline:
            # Add slide
            if slide_data['slide_number'] == 1:
                slide_layout = prs.slide_layouts[6]  # Blank
            else:
                slide_layout = prs.slide_layouts[6]  # Blank
            
            slide = prs.slides.add_slide(slide_layout)
            
            # Fetch and add background image
            try:
                img_data = self.fetch_image(slide_data['image_keyword'])
                img_stream = BytesIO(img_data)
                
                # Add image as background
                left = top = Inches(0)
                pic = slide.shapes.add_picture(img_stream, left, top, 
                                              width=prs.slide_width, 
                                              height=prs.slide_height)
                
                # Send to back
                slide.shapes._spTree.remove(pic._element)
                slide.shapes._spTree.insert(2, pic._element)
            except Exception as e:
                print(f"Error adding image: {e}")
            
            # Add semi-transparent overlay box for text
            left = Inches(0.5)
            top = Inches(1)
            width = Inches(9)
            height = Inches(5.5)
            
            shape = slide.shapes.add_shape(1, left, top, width, height)  # Rectangle
            shape.fill.solid()
            shape.fill.fore_color.rgb = RGBColor(255, 255, 255)
            shape.fill.fore_color.brightness = 0.8
            shape.line.fill.background()
            
            # Add title
            left = Inches(1)
            top = Inches(1.5)
            width = Inches(8)
            height = Inches(1)
            
            title_box = slide.shapes.add_textbox(left, top, width, height)
            title_frame = title_box.text_frame
            title_frame.text = slide_data['title']
            
            title_para = title_frame.paragraphs[0]
            title_para.font.size = Pt(44)
            title_para.font.bold = True
            title_para.font.color.rgb = RGBColor(31, 78, 121)
            title_para.alignment = PP_ALIGN.CENTER
            
            # Add bullets
            if slide_data.get('bullets'):
                left = Inches(1.5)
                top = Inches(3)
                width = Inches(7)
                height = Inches(3.5)
                
                text_box = slide.shapes.add_textbox(left, top, width, height)
                text_frame = text_box.text_frame
                text_frame.word_wrap = True
                
                for bullet in slide_data['bullets']:
                    p = text_frame.add_paragraph()
                    p.text = bullet
                    p.font.size = Pt(24)
                    p.font.color.rgb = RGBColor(0, 0, 0)
                    p.level = 0
            
            # Add speaker notes
            if slide_data.get('speaker_notes'):
                notes_slide = slide.notes_slide
                text_frame = notes_slide.notes_text_frame
                text_frame.text = slide_data['speaker_notes']
            
            # Add watermark
            if add_watermark:
                left = Inches(8.5)
                top = Inches(6.8)
                width = Inches(1.2)
                height = Inches(0.5)
                
                watermark = slide.shapes.add_textbox(left, top, width, height)
                watermark_frame = watermark.text_frame
                watermark_frame.text = "ToolsHub"
                
                p = watermark_frame.paragraphs[0]
                p.font.size = Pt(10)
                p.font.color.rgb = RGBColor(128, 128, 128)
                p.alignment = PP_ALIGN.RIGHT
        
        # Save
        filename = f"presentation_{datetime.now().strftime('%Y%m%d_%H%M%S')}.pptx"
        filepath = os.path.join(self.temp_dir, filename)
        prs.save(filepath)
        
        return filepath
    
    def generate_pdf(self, outline: List[Dict], topic: str, add_watermark: bool = False) -> str:
        """Generate PDF presentation"""
        
        filename = f"presentation_{datetime.now().strftime('%Y%m%d_%H%M%S')}.pdf"
        filepath = os.path.join(self.temp_dir, filename)
        
        doc = SimpleDocTemplate(filepath, pagesize=landscape(letter))
        story = []
        styles = getSampleStyleSheet()
        
        # Custom styles
        title_style = ParagraphStyle(
            'CustomTitle',
            parent=styles['Heading1'],
            fontSize=36,
            textColor='#1f4e79',
            alignment=TA_CENTER,
            spaceAfter=30
        )
        
        bullet_style = ParagraphStyle(
            'CustomBullet',
            parent=styles['BodyText'],
            fontSize=18,
            leftIndent=20,
            spaceAfter=12
        )
        
        for i, slide_data in enumerate(outline):
            # Title
            story.append(Paragraph(slide_data['title'], title_style))
            story.append(Spacer(1, 0.3*inch))
            
            # Bullets
            for bullet in slide_data.get('bullets', []):
                story.append(Paragraph(f"• {bullet}", bullet_style))
            
            story.append(Spacer(1, 0.3*inch))
            
            # Speaker notes
            if slide_data.get('speaker_notes'):
                notes_style = ParagraphStyle(
                    'Notes',
                    parent=styles['Italic'],
                    fontSize=12,
                    textColor='#666666'
                )
                story.append(Paragraph(f"<i>Notes: {slide_data['speaker_notes']}</i>", notes_style))
            
            # Watermark
            if add_watermark:
                watermark_style = ParagraphStyle(
                    'Watermark',
                    parent=styles['Normal'],
                    fontSize=8,
                    textColor='#999999',
                    alignment=TA_CENTER
                )
                story.append(Spacer(1, 0.2*inch))
                story.append(Paragraph("Generated by ToolsHub", watermark_style))
            
            # Page break except for last slide
            if i < len(outline) - 1:
                story.append(PageBreak())
        
        doc.build(story)
        return filepath
    
    def generate_docx(self, outline: List[Dict], topic: str, add_watermark: bool = False) -> str:
        """Generate Word document presentation"""
        
        doc = Document()
        
        # Set default font
        style = doc.styles['Normal']
        font = style.font
        font.name = 'Poppins'
        font.size = DocxPt(12)
        
        for i, slide_data in enumerate(outline):
            # Title
            title = doc.add_heading(slide_data['title'], level=1)
            title.alignment = WD_ALIGN_PARAGRAPH.CENTER
            
            # Bullets
            for bullet in slide_data.get('bullets', []):
                doc.add_paragraph(bullet, style='List Bullet')
            
            # Speaker notes
            if slide_data.get('speaker_notes'):
                doc.add_paragraph()
                notes = doc.add_paragraph()
                notes.add_run('Speaker Notes: ').bold = True
                notes.add_run(slide_data['speaker_notes'])
                notes_format = notes.paragraph_format
                notes_format.left_indent = DocxInches(0.5)
            
            # Watermark
            if add_watermark:
                watermark = doc.add_paragraph()
                watermark_run = watermark.add_run('Generated by ToolsHub')
                watermark_run.font.size = DocxPt(8)
                watermark_run.font.color.rgb = DocxRGB(150, 150, 150)
                watermark.alignment = WD_ALIGN_PARAGRAPH.RIGHT
            
            # Page break except for last slide
            if i < len(outline) - 1:
                doc.add_page_break()
        
        # Save
        filename = f"presentation_{datetime.now().strftime('%Y%m%d_%H%M%S')}.docx"
        filepath = os.path.join(self.temp_dir, filename)
        doc.save(filepath)
        
        return filepath


def generate_presentation(topic: str, slide_count: int, format: str = 'pptx', 
                         add_watermark: bool = True) -> Dict[str, Any]:
    """
    Main function to generate presentation
    
    Args:
        topic: Presentation topic
        slide_count: Number of slides (3-20)
        format: Output format (pptx, pdf, docx)
        add_watermark: Add ToolsHub watermark
    
    Returns:
        {
            'success': bool,
            'outline': List[Dict],
            'file_path': str,
            'message': str
        }
    """
    
    try:
        # Validate inputs
        if not topic or len(topic.strip()) < 3:
            return {
                'success': False,
                'message': 'Topic must be at least 3 characters long'
            }
        
        if not isinstance(slide_count, int) or slide_count < 3 or slide_count > 20:
            return {
                'success': False,
                'message': 'Slide count must be between 3 and 20'
            }
        
        if format not in ['pptx', 'pdf', 'docx']:
            return {
                'success': False,
                'message': 'Format must be pptx, pdf, or docx'
            }
        
        # Generate presentation
        generator = PresentationGenerator()
        
        # Step 1: Generate outline
        outline = generator.generate_outline(topic, slide_count)
        
        # Step 2: Generate file
        if format == 'pptx':
            filepath = generator.generate_pptx(outline, topic, add_watermark)
        elif format == 'pdf':
            filepath = generator.generate_pdf(outline, topic, add_watermark)
        else:  # docx
            filepath = generator.generate_docx(outline, topic, add_watermark)
        
        return {
            'success': True,
            'outline': outline,
            'file_path': filepath,
            'message': f'Presentation generated successfully as {format.upper()}'
        }
        
    except Exception as e:
        return {
            'success': False,
            'message': f'Error generating presentation: {str(e)}'
        }


if __name__ == '__main__':
    # Test
    result = generate_presentation(
        topic="Artificial Intelligence in Healthcare",
        slide_count=5,
        format='pptx',
        add_watermark=True
    )
    print(json.dumps(result, indent=2))
