"""
PDF Page Cleaner - Remove or keep selected pages
"""

from PyPDF2 import PdfReader, PdfWriter
import os

def clean_pdf_pages(input_path, pages_to_keep):
    """
    Create new PDF with only selected pages
    
    Args:
        input_path: Path to input PDF
        pages_to_keep: List of page numbers (1-indexed)
    
    Returns:
        Path to cleaned PDF
    """
    try:
        # Read PDF
        reader = PdfReader(input_path)
        writer = PdfWriter()
        
        total_pages = len(reader.pages)
        
        # Validate page numbers
        for page_num in pages_to_keep:
            if page_num < 1 or page_num > total_pages:
                raise ValueError(f"Page {page_num} out of range (1-{total_pages})")
        
        # Add selected pages (convert to 0-indexed)
        for page_num in sorted(set(pages_to_keep)):
            writer.add_page(reader.pages[page_num - 1])
        
        # Create output path
        output_path = input_path.replace('uploads', 'outputs')
        output_path = output_path.rsplit('.', 1)[0] + '_cleaned.pdf'
        
        # Ensure output directory exists
        os.makedirs(os.path.dirname(output_path), exist_ok=True)
        
        # Write output PDF
        with open(output_path, 'wb') as output_file:
            writer.write(output_file)
        
        return output_path
        
    except Exception as e:
        raise Exception(f"PDF cleaning failed: {str(e)}")
