"""
Logo Analyzer Tool - Analyzes actual image properties
Uses: ColorThief for color extraction, PIL for image analysis, OpenCV for shape detection
"""

from PIL import Image, ImageStat
import colorthief
import webcolors
import cv2
import numpy as np
from collections import Counter
import pytesseract
import os

def rgb_to_hex(rgb):
    """Convert RGB tuple to hex color"""
    return '#{:02x}{:02x}{:02x}'.format(rgb[0], rgb[1], rgb[2])

def get_color_name(rgb):
    """Get closest color name"""
    try:
        color_name = webcolors.rgb_to_name(rgb)
        return color_name.capitalize()
    except ValueError:
        # Find closest named color
        min_dist = float('inf')
        closest_name = 'Unknown'
        for name, hex_val in webcolors.CSS3_HEX_TO_NAMES.items():
            r, g, b = webcolors.hex_to_rgb(hex_val)
            dist = sum([(a - b) ** 2 for a, b in zip(rgb, (r, g, b))])
            if dist < min_dist:
                min_dist = dist
                closest_name = webcolors.CSS3_HEX_TO_NAMES[hex_val]
        return closest_name.capitalize()

def analyze_colors(image_path):
    """Extract dominant colors and analyze color harmony"""
    try:
        color_thief = colorthief.ColorThief(image_path)
        
        # Get dominant color
        dominant_color = color_thief.get_color(quality=1)
        
        # Get color palette (top 5 colors)
        palette = color_thief.get_palette(color_count=6, quality=1)[:5]
        
        colors_analysis = []
        for rgb in palette:
            hex_code = rgb_to_hex(rgb)
            color_name = get_color_name(rgb)
            colors_analysis.append({
                'rgb': rgb,
                'hex': hex_code,
                'name': color_name
            })
        
        # Analyze color harmony
        harmony = analyze_color_harmony(palette)
        
        return {
            'dominant_color': {
                'rgb': dominant_color,
                'hex': rgb_to_hex(dominant_color),
                'name': get_color_name(dominant_color)
            },
            'palette': colors_analysis,
            'harmony': harmony
        }
    except Exception as e:
        return {
            'dominant_color': {'hex': '#000000', 'name': 'Unknown'},
            'palette': [],
            'harmony': 'Unable to analyze'
        }

def analyze_color_harmony(palette):
    """Analyze color relationships"""
    if len(palette) < 2:
        return "Monochromatic - Uses variations of a single color"
    
    # Simple hue analysis
    hues = []
    for rgb in palette:
        r, g, b = [x / 255.0 for x in rgb]
        max_c = max(r, g, b)
        min_c = min(r, g, b)
        
        if max_c == min_c:
            hue = 0
        elif max_c == r:
            hue = 60 * (((g - b) / (max_c - min_c)) % 6)
        elif max_c == g:
            hue = 60 * (((b - r) / (max_c - min_c)) + 2)
        else:
            hue = 60 * (((r - g) / (max_c - min_c)) + 4)
        
        hues.append(hue)
    
    # Check relationships
    hue_diff = abs(hues[0] - hues[1]) if len(hues) > 1 else 0
    
    if hue_diff < 30:
        return "Analogous - Colors are adjacent on color wheel (harmonious)"
    elif 150 < hue_diff < 210:
        return "Complementary - Opposite colors creating strong contrast"
    elif 60 < hue_diff < 90 or 270 < hue_diff < 300:
        return "Triadic - Evenly spaced colors (vibrant and balanced)"
    else:
        return "Custom scheme - Unique color combination"

def analyze_layout_and_shapes(image_path):
    """Analyze logo layout, balance, and complexity"""
    try:
        img = cv2.imread(image_path)
        gray = cv2.cvtColor(img, cv2.COLOR_BGR2GRAY)
        
        # Detect edges
        edges = cv2.Canny(gray, 50, 150)
        
        # Calculate complexity (edge density)
        edge_density = np.sum(edges > 0) / edges.size
        
        if edge_density < 0.05:
            complexity = "Simple - Clean and minimalist design"
        elif edge_density < 0.15:
            complexity = "Moderate - Balanced detail level"
        else:
            complexity = "Complex - Intricate with many details"
        
        # Analyze symmetry/balance
        height, width = gray.shape
        left_half = gray[:, :width//2]
        right_half = cv2.flip(gray[:, width//2:], 1)
        
        # Resize to match if needed
        if left_half.shape != right_half.shape:
            right_half = cv2.resize(right_half, (left_half.shape[1], left_half.shape[0]))
        
        similarity = np.sum(left_half == right_half) / left_half.size
        
        if similarity > 0.8:
            balance = "Highly Symmetric - Strong bilateral balance"
        elif similarity > 0.5:
            balance = "Moderately Balanced - Some asymmetry for interest"
        else:
            balance = "Asymmetric - Dynamic and unconventional layout"
        
        # Detect shapes
        contours, _ = cv2.findContours(edges, cv2.RETR_EXTERNAL, cv2.CHAIN_APPROX_SIMPLE)
        shape_count = len([c for c in contours if cv2.contourArea(c) > 100])
        
        return {
            'complexity': complexity,
            'balance': balance,
            'shape_count': shape_count,
            'layout_note': f"Contains approximately {shape_count} distinct visual elements"
        }
    except Exception as e:
        return {
            'complexity': 'Unable to analyze',
            'balance': 'Unable to determine',
            'shape_count': 0,
            'layout_note': 'Layout analysis unavailable'
        }

def analyze_text_and_font(image_path):
    """Detect text in logo and analyze font style"""
    try:
        img = Image.open(image_path)
        
        # Extract text using OCR
        text = pytesseract.image_to_string(img, config='--psm 6')
        text = text.strip()
        
        if not text:
            return {
                'has_text': False,
                'font_style': 'Icon/Symbol Only - No text detected',
                'readability': 'N/A - No text present'
            }
        
        # Analyze text characteristics
        has_uppercase = any(c.isupper() for c in text)
        has_lowercase = any(c.islower() for c in text)
        has_numbers = any(c.isdigit() for c in text)
        
        # Infer font style (basic heuristics)
        if has_uppercase and not has_lowercase:
            font_style = "Bold Uppercase - Strong, authoritative feel (likely Sans-serif)"
        elif has_lowercase and has_uppercase:
            font_style = "Mixed Case - Professional and readable (likely Serif or Sans-serif)"
        else:
            font_style = "Custom Styling - Unique typography"
        
        # Readability score
        text_length = len(text)
        if text_length < 10:
            readability = "Excellent - Short and memorable"
        elif text_length < 20:
            readability = "Good - Clear and concise"
        else:
            readability = "Consider simplifying - Long text may reduce impact"
        
        return {
            'has_text': True,
            'detected_text': text[:50],  # Limit length
            'font_style': font_style,
            'readability': readability
        }
    except Exception as e:
        return {
            'has_text': False,
            'font_style': 'Unable to detect text',
            'readability': 'Analysis unavailable'
        }

def analyze_contrast_and_accessibility(image_path):
    """Analyze color contrast and accessibility"""
    try:
        img = Image.open(image_path).convert('RGB')
        stat = ImageStat.Stat(img)
        
        # Average brightness
        avg_brightness = sum(stat.mean) / 3
        
        # Calculate contrast
        r_range = stat.extrema[0][1] - stat.extrema[0][0]
        g_range = stat.extrema[1][1] - stat.extrema[1][0]
        b_range = stat.extrema[2][1] - stat.extrema[2][0]
        avg_range = (r_range + g_range + b_range) / 3
        
        if avg_range > 200:
            contrast = "High Contrast - Excellent visibility and clarity"
        elif avg_range > 100:
            contrast = "Moderate Contrast - Good balance"
        else:
            contrast = "Low Contrast - May reduce legibility"
        
        # Accessibility notes
        if avg_brightness < 100:
            accessibility = "Dark theme - Ensure sufficient contrast with light backgrounds"
        elif avg_brightness > 200:
            accessibility = "Light theme - Works well on dark backgrounds"
        else:
            accessibility = "Balanced brightness - Versatile across contexts"
        
        return {
            'contrast': contrast,
            'accessibility': accessibility,
            'avg_brightness': round(avg_brightness, 1)
        }
    except Exception as e:
        return {
            'contrast': 'Unable to analyze',
            'accessibility': 'Analysis unavailable',
            'avg_brightness': 0
        }

def analyze_professional_impression(colors, layout, text_info):
    """Determine professional impression and industry fit"""
    impressions = []
    
    # Based on colors
    if colors.get('dominant_color', {}).get('name', '').lower() in ['blue', 'navy', 'darkblue']:
        impressions.append("Corporate/Professional - Blue conveys trust and stability")
    elif colors.get('dominant_color', {}).get('name', '').lower() in ['red', 'crimson']:
        impressions.append("Bold/Energetic - Red suggests passion and action")
    elif colors.get('dominant_color', {}).get('name', '').lower() in ['green', 'lime']:
        impressions.append("Eco-friendly/Growth - Green represents nature and prosperity")
    elif colors.get('dominant_color', {}).get('name', '').lower() in ['black', 'gray', 'silver']:
        impressions.append("Minimalist/Luxury - Neutral tones suggest sophistication")
    else:
        impressions.append("Creative/Unique - Distinctive color choice stands out")
    
    # Based on complexity
    if 'Simple' in layout.get('complexity', ''):
        impressions.append("Modern/Clean - Minimalist approach aligns with contemporary trends")
    elif 'Complex' in layout.get('complexity', ''):
        impressions.append("Detailed/Ornate - Rich design may suit traditional industries")
    
    # Based on text
    if not text_info.get('has_text'):
        impressions.append("Icon-focused - Strong visual identity without relying on text")
    
    return ' | '.join(impressions)

def analyze_logo(image_path):
    """
    Main analysis function - returns comprehensive logo analysis
    All analysis based on actual image properties
    """
    try:
        # Perform all analyses
        colors = analyze_colors(image_path)
        layout = analyze_layout_and_shapes(image_path)
        text_info = analyze_text_and_font(image_path)
        contrast = analyze_contrast_and_accessibility(image_path)
        impression = analyze_professional_impression(colors, layout, text_info)
        
        # Compile comprehensive analysis
        analysis = {
            'colors': colors,
            'layout': layout,
            'typography': text_info,
            'contrast_accessibility': contrast,
            'professional_impression': impression,
            'recommendations': generate_recommendations(colors, layout, text_info, contrast)
        }
        
        return analysis
        
    except Exception as e:
        raise Exception(f"Logo analysis failed: {str(e)}")

def generate_recommendations(colors, layout, text_info, contrast):
    """Generate actionable recommendations"""
    recs = []
    
    # Color recommendations
    if len(colors.get('palette', [])) > 4:
        recs.append("Consider simplifying color palette to 2-3 main colors for better brand consistency")
    
    # Layout recommendations
    if 'Complex' in layout.get('complexity', ''):
        recs.append("Simplify design for better scalability - complex logos may lose detail at small sizes")
    
    # Typography recommendations
    if text_info.get('has_text') and len(text_info.get('detected_text', '')) > 15:
        recs.append("Shorter text improves memorability - consider using initials or abbreviation")
    
    # Contrast recommendations
    if 'Low' in contrast.get('contrast', ''):
        recs.append("Increase color contrast for better visibility across different media")
    
    if not recs:
        recs.append("Logo shows strong design principles - well-balanced and professional")
    
    return recs

# For testing
if __name__ == "__main__":
    pass
