"""
Social Media Caption Rewriter
Generates 3 unique captions tailored to platform and tone
"""

import re
from collections import Counter

# Emoji library for different tones
EMOJIS = {
    'funny': ['😂', '🤣', '😜', '🙃', '😅', '🤪', '😎', '🎉'],
    'emotional': ['❤️', '💕', '🥺', '😊', '🌟', '💫', '✨', '🙏'],
    'professional': ['💼', '📊', '🎯', '📈', '💡', '✅', '🔑', '👔'],
    'inspiring': ['🌟', '💪', '🚀', '⭐', '🌈', '🔥', '💫', '✨']
}

def extract_keywords(caption):
    """Extract keywords from caption for hashtag generation"""
    words = re.findall(r'\b[a-zA-Z]{3,}\b', caption.lower())
    stop_words = {'the', 'a', 'an', 'and', 'or', 'but', 'in', 'on', 'at', 'to', 'for', 'of', 'with'}
    keywords = [w for w in words if w not in stop_words]
    return keywords[:5]

def generate_hashtags(caption, platform, count=10):
    """Generate relevant hashtags from caption keywords"""
    keywords = extract_keywords(caption)
    
    hashtags = []
    for keyword in keywords:
        hashtags.append(f"#{keyword}")
        # Add variations
        if len(keyword) > 5:
            hashtags.append(f"#{keyword}s")
            hashtags.append(f"#{keyword}life")
    
    # Add platform-specific trending tags
    if platform == 'instagram':
        hashtags.extend(['#instagood', '#photooftheday', '#instagram'])
    elif platform == 'tiktok':
        hashtags.extend(['#fyp', '#foryou', '#viral'])
    elif platform == 'linkedin':
        hashtags.extend(['#professional', '#career', '#business'])
    elif platform == 'facebook':
        hashtags.extend(['#socialmedia', '#community'])
    
    return hashtags[:count]

def apply_tone(text, tone, emojis_list):
    """Apply tone modifications to text"""
    if tone == 'funny':
        # Add humor indicators
        text += " 😂"
        if "!" not in text[-3:]:
            text = text.rstrip('.') + "!"
    
    elif tone == 'emotional':
        # Add emotional depth
        text += " ❤️"
        if "." in text:
            text = text.replace(".", "...")
    
    elif tone == 'professional':
        # Keep formal
        text = text.replace("!", ".")
        text = text.replace("...", ".")
    
    elif tone == 'inspiring':
        # Add motivational feel
        text += " 💪"
        if "!" not in text:
            text = text.rstrip('.') + "!"
    
    return text

def format_for_platform(caption, hashtags, platform):
    """Format caption according to platform requirements"""
    
    if platform == 'instagram':
        # Line breaks + hashtags
        formatted = caption + "\n.\n.\n.\n"
        formatted += " ".join(hashtags[:15])
        
    elif platform == 'twitter' or platform == 'x':
        # Keep under 280 characters
        hashtags_str = " ".join(hashtags[:2])
        total = caption + " " + hashtags_str
        if len(total) > 280:
            caption = caption[:270-len(hashtags_str)]
        formatted = caption + " " + hashtags_str
        
    elif platform == 'tiktok':
        # Short with trending hashtags
        formatted = caption + "\n\n"
        formatted += " ".join(hashtags[:8])
        
    elif platform == 'facebook':
        # Longer form with fewer hashtags
        formatted = caption + "\n\n"
        formatted += " ".join(hashtags[:3])
        
    elif platform == 'linkedin':
        # Professional with strategic hashtags
        formatted = caption + "\n\n"
        formatted += " ".join(hashtags[:5])
    
    else:
        formatted = caption + "\n\n" + " ".join(hashtags[:10])
    
    return formatted

def rewrite_caption_variation(original, tone, variation_type):
    """Create a variation of the caption"""
    
    # Extract core message
    words = original.split()
    
    if variation_type == 1:
        # Question format
        if '?' not in original:
            rewritten = "Have you ever wondered about " + original.lower() + "?"
        else:
            rewritten = original
    
    elif variation_type == 2:
        # Statement + CTA
        rewritten = original.rstrip('.!?') + ". "
        if tone == 'funny':
            rewritten += "Tag someone who needs to see this! 😂"
        elif tone == 'emotional':
            rewritten += "Double tap if you agree ❤️"
        elif tone == 'professional':
            rewritten += "Share your thoughts in the comments."
        elif tone == 'inspiring':
            rewritten += "Let's make it happen! 💪"
        else:
            rewritten += "What do you think? 🤔"
    
    elif variation_type == 3:
        # Story format
        if tone == 'funny':
            rewritten = "POV: " + original.lower() + " 🤣"
        elif tone == 'emotional':
            rewritten = "This is why " + original.lower() + " matters ❤️"
        elif tone == 'professional':
            rewritten = "Key insight: " + original
        elif tone == 'inspiring':
            rewritten = "Remember: " + original + " 🌟"
        else:
            rewritten = "Here's the thing: " + original
    
    else:
        rewritten = original
    
    return rewritten

def rewrite_captions(caption, platform, tone):
    """
    Generate 3 unique captions tailored to platform and tone
    
    Args:
        caption: Original caption text
        platform: instagram/facebook/twitter/tiktok/linkedin
        tone: funny/emotional/professional/inspiring/engaging
    
    Returns:
        List of 3 unique formatted captions
    """
    try:
        # Generate hashtags from caption
        hashtags = generate_hashtags(caption, platform)
        
        # Get emojis for tone
        emojis = EMOJIS.get(tone, EMOJIS['funny'])
        
        # Generate 3 variations
        captions = []
        
        for i in range(1, 4):
            # Create variation
            rewritten = rewrite_caption_variation(caption, tone, i)
            
            # Apply tone
            rewritten = apply_tone(rewritten, tone, emojis)
            
            # Format for platform
            formatted = format_for_platform(rewritten, hashtags, platform)
            
            captions.append(formatted)
        
        return captions
        
    except Exception as e:
        # Fallback captions
        return [
            f"{caption} {hashtags[0] if hashtags else '#content'}",
            f"{caption} What do you think? {hashtags[1] if len(hashtags) > 1 else '#social'}",
            f"{caption} Share this! {hashtags[2] if len(hashtags) > 2 else '#media'}"
        ]

# For testing
if __name__ == "__main__":
    test_caption = "Check out our new product launch! Amazing features and great design."
    captions = rewrite_captions(test_caption, 'instagram', 'exciting')
    for i, cap in enumerate(captions, 1):
        print(f"\n=== Caption {i} ===")
        print(cap)
