"""
AI Assignment/Research Paper Generator - Production Version
Generates complete academic assignments with proper structure
Exports to DOCX and PDF formats
Uses free AI models for content generation
"""

import os
import json
from datetime import datetime
from typing import Dict, List, Any
import re

# For DOCX generation
from docx import Document
from docx.shared import Inches, Pt, RGBColor
from docx.enum.text import WD_ALIGN_PARAGRAPH
from docx.enum.style import WD_STYLE_TYPE

# For PDF generation
from reportlab.lib.pagesizes import letter
from reportlab.lib.units import inch
from reportlab.platypus import SimpleDocTemplate, Paragraph, Spacer, PageBreak, Table, TableStyle
from reportlab.lib.styles import getSampleStyleSheet, ParagraphStyle
from reportlab.lib.enums import TA_CENTER, TA_JUSTIFY, TA_LEFT
from reportlab.lib import colors

# AI Generation
import google.generativeai as genai

# Environment
GEMINI_API_KEY = os.getenv('GEMINI_API_KEY', '')


class AssignmentGenerator:
    """Generate complete academic assignments with AI"""
    
    def __init__(self):
        self.temp_dir = '/tmp/assignments'
        os.makedirs(self.temp_dir, exist_ok=True)
        
        # Initialize AI
        if GEMINI_API_KEY:
            genai.configure(api_key=GEMINI_API_KEY)
            self.model = genai.GenerativeModel('gemini-pro')
        else:
            self.model = None
    
    def generate_toc(self, topic: str, page_count: int) -> List[Dict[str, Any]]:
        """Generate Table of Contents structure"""
        
        if not self.model:
            return self._generate_template_toc(topic, page_count)
        
        # Calculate chapter distribution (excluding title page, TOC, intro, conclusion, references)
        content_pages = max(1, page_count - 5)
        chapters = min(max(3, content_pages // 2), 8)
        
        prompt = f"""Create a detailed Table of Contents for a {page_count}-page academic assignment on: {topic}

Structure:
1. Title Page
2. Table of Contents
3. Introduction (1 page)
4. {chapters} main chapters/sections (distribute remaining pages)
5. Conclusion (1 page)
6. References (1 page)

For each chapter, provide:
- Chapter number and title
- Brief description of what will be covered
- Estimated word count (300-350 words per page)

Format as JSON:
{{
  "title": "Assignment Title",
  "sections": [
    {{
      "type": "chapter",
      "number": 1,
      "title": "Chapter Title",
      "description": "What this chapter covers",
      "pages": 2,
      "word_count": 650
    }},
    ...
  ]
}}

Make it academic, well-structured, and comprehensive."""

        try:
            response = self.model.generate_content(prompt)
            text = response.text.strip()
            
            # Extract JSON
            if '```json' in text:
                text = text.split('```json')[1].split('```')[0].strip()
            elif '```' in text:
                text = text.split('```')[1].split('```')[0].strip()
            
            toc_data = json.loads(text)
            return toc_data
            
        except Exception as e:
            print(f"AI TOC generation error: {e}")
            return self._generate_template_toc(topic, page_count)
    
    def _generate_template_toc(self, topic: str, page_count: int) -> Dict[str, Any]:
        """Fallback template-based TOC"""
        
        content_pages = max(1, page_count - 5)
        chapters = min(max(3, content_pages // 2), 6)
        pages_per_chapter = content_pages // chapters
        
        sections = [
            {
                "type": "intro",
                "number": None,
                "title": "Introduction",
                "description": f"Overview of {topic} and scope of this assignment",
                "pages": 1,
                "word_count": 325
            }
        ]
        
        for i in range(chapters):
            sections.append({
                "type": "chapter",
                "number": i + 1,
                "title": f"Chapter {i+1}: Key Aspects of {topic}",
                "description": f"Detailed analysis of aspect {i+1}",
                "pages": pages_per_chapter,
                "word_count": pages_per_chapter * 325
            })
        
        sections.extend([
            {
                "type": "conclusion",
                "number": None,
                "title": "Conclusion",
                "description": f"Summary and final thoughts on {topic}",
                "pages": 1,
                "word_count": 325
            },
            {
                "type": "references",
                "number": None,
                "title": "References",
                "description": "APA format citations",
                "pages": 1,
                "word_count": 0
            }
        ])
        
        return {
            "title": f"A Comprehensive Study on {topic}",
            "sections": sections
        }
    
    def generate_content(self, section: Dict[str, Any], topic: str) -> str:
        """Generate content for a section"""
        
        if not self.model:
            return self._generate_template_content(section, topic)
        
        word_count = section.get('word_count', 325)
        section_title = section.get('title', 'Section')
        section_desc = section.get('description', '')
        
        prompt = f"""Write {word_count} words of academic content for this section:

Topic: {topic}
Section: {section_title}
Description: {section_desc}

Requirements:
- Formal academic tone
- Well-structured paragraphs
- Clear arguments and explanations
- Plagiarism-free original content
- Professional language
- Proper flow and transitions

Write ONLY the content, no title or metadata."""

        try:
            response = self.model.generate_content(prompt)
            content = response.text.strip()
            
            # Ensure minimum length
            words = content.split()
            if len(words) < word_count * 0.8:
                # Generate more
                additional_prompt = f"Continue this content to reach {word_count} words:\n\n{content}"
                additional_response = self.model.generate_content(additional_prompt)
                content += "\n\n" + additional_response.text.strip()
            
            return content
            
        except Exception as e:
            print(f"Content generation error: {e}")
            return self._generate_template_content(section, topic)
    
    def _generate_template_content(self, section: Dict[str, Any], topic: str) -> str:
        """Fallback template content"""
        
        word_count = section.get('word_count', 325)
        section_title = section.get('title', 'Section')
        
        # Generate placeholder academic content
        base_text = f"""This section provides a comprehensive analysis of {section_title} in the context of {topic}. 
        
The study examines various aspects and dimensions that are crucial to understanding this subject matter. Through careful research and analysis, we explore the fundamental concepts, theories, and practical applications relevant to this area.

Current research in this field demonstrates significant developments and ongoing debates among scholars. Various perspectives contribute to our understanding, and it is essential to consider multiple viewpoints when examining complex topics such as this one.

The methodology employed in examining these aspects includes both theoretical frameworks and empirical evidence. This approach ensures a balanced and thorough investigation of the subject matter, providing readers with valuable insights and practical knowledge.

Furthermore, the implications of these findings extend beyond theoretical considerations. They offer practical applications and suggest directions for future research and development in this important area of study."""
        
        # Expand to meet word count
        words = base_text.split()
        while len(words) < word_count:
            base_text += f"\n\nAdditional analysis reveals further important considerations regarding {topic}. These factors play a crucial role in shaping our understanding and provide context for the broader implications of this research."
            words = base_text.split()
        
        return ' '.join(words[:word_count])
    
    def generate_references(self, topic: str, count: int = 10) -> List[str]:
        """Generate APA format references"""
        
        if not self.model:
            return self._generate_template_references(topic, count)
        
        prompt = f"""Generate {count} realistic APA format references for an academic paper on: {topic}

Requirements:
- Proper APA 7th edition format
- Mix of books, journal articles, and websites
- Recent years (2018-2024)
- Realistic author names and titles
- Alphabetically ordered by author last name

Format each reference properly with hanging indent."""

        try:
            response = self.model.generate_content(prompt)
            refs_text = response.text.strip()
            
            # Parse references
            references = [ref.strip() for ref in refs_text.split('\n') if ref.strip() and not ref.strip().startswith('#')]
            
            return references[:count]
            
        except Exception as e:
            print(f"References generation error: {e}")
            return self._generate_template_references(topic, count)
    
    def _generate_template_references(self, topic: str, count: int) -> List[str]:
        """Fallback template references"""
        
        current_year = datetime.now().year
        references = []
        
        for i in range(count):
            year = current_year - (i % 5)
            author_last = ['Smith', 'Johnson', 'Williams', 'Brown', 'Jones', 'Garcia', 'Miller', 'Davis'][i % 8]
            author_first = ['J.', 'M.', 'A.', 'R.', 'S.', 'T.', 'K.', 'L.'][i % 8]
            
            if i % 3 == 0:
                # Book
                ref = f"{author_last}, {author_first} ({year}). Understanding {topic}: A comprehensive guide. Academic Press."
            elif i % 3 == 1:
                # Journal
                ref = f"{author_last}, {author_first} ({year}). Research on {topic}. Journal of Advanced Studies, {20+i}({1+i%4}), {100+i*10}-{120+i*10}."
            else:
                # Website
                ref = f"{author_last}, {author_first} ({year}). {topic} in modern context. Retrieved from https://example-academic-source.edu/research"
            
            references.append(ref)
        
        return sorted(references)
    
    def generate_docx(self, toc_data: Dict, topic: str) -> str:
        """Generate complete DOCX assignment"""
        
        doc = Document()
        
        # Set default styles
        style = doc.styles['Normal']
        font = style.font
        font.name = 'Times New Roman'
        font.size = Pt(12)
        
        # Title Page
        title_para = doc.add_paragraph()
        title_run = title_para.add_run(toc_data['title'])
        title_run.font.size = Pt(24)
        title_run.font.bold = True
        title_para.alignment = WD_ALIGN_PARAGRAPH.CENTER
        
        doc.add_paragraph()
        doc.add_paragraph()
        
        # Author and date
        author_para = doc.add_paragraph()
        author_para.add_run('Student Name\nCourse Name\nInstructor Name')
        author_para.alignment = WD_ALIGN_PARAGRAPH.CENTER
        
        doc.add_paragraph()
        date_para = doc.add_paragraph()
        date_para.add_run(datetime.now().strftime('%B %d, %Y'))
        date_para.alignment = WD_ALIGN_PARAGRAPH.CENTER
        
        doc.add_page_break()
        
        # Table of Contents
        toc_heading = doc.add_heading('Table of Contents', level=1)
        toc_heading.alignment = WD_ALIGN_PARAGRAPH.CENTER
        
        page_num = 3
        for section in toc_data['sections']:
            if section['type'] == 'chapter':
                toc_entry = doc.add_paragraph()
                toc_entry.add_run(f"Chapter {section['number']}: {section['title']}").bold = True
                toc_entry.add_run(f" {'.' * 50} {page_num}")
                page_num += section['pages']
            else:
                toc_entry = doc.add_paragraph()
                toc_entry.add_run(f"{section['title']}").bold = True
                toc_entry.add_run(f" {'.' * 50} {page_num}")
                page_num += section['pages']
        
        doc.add_page_break()
        
        # Generate content for each section
        for section in toc_data['sections']:
            # Section heading
            if section['type'] == 'chapter':
                heading = doc.add_heading(f"Chapter {section['number']}: {section['title']}", level=1)
            else:
                heading = doc.add_heading(section['title'], level=1)
            
            # Generate and add content
            if section['type'] != 'references':
                content = self.generate_content(section, topic)
                paragraphs = content.split('\n\n')
                
                for para_text in paragraphs:
                    if para_text.strip():
                        para = doc.add_paragraph(para_text.strip())
                        para.alignment = WD_ALIGN_PARAGRAPH.JUSTIFY
                        para_format = para.paragraph_format
                        para_format.line_spacing = 2.0
                        para_format.space_after = Pt(12)
            else:
                # Add references
                references = self.generate_references(topic)
                for ref in references:
                    ref_para = doc.add_paragraph(ref)
                    ref_format = ref_para.paragraph_format
                    ref_format.left_indent = Inches(0.5)
                    ref_format.first_line_indent = Inches(-0.5)
            
            # Page break except for last section
            if section != toc_data['sections'][-1]:
                doc.add_page_break()
        
        # Save
        filename = f"assignment_{datetime.now().strftime('%Y%m%d_%H%M%S')}.docx"
        filepath = os.path.join(self.temp_dir, filename)
        doc.save(filepath)
        
        return filepath
    
    def generate_pdf(self, toc_data: Dict, topic: str) -> str:
        """Generate complete PDF assignment"""
        
        filename = f"assignment_{datetime.now().strftime('%Y%m%d_%H%M%S')}.pdf"
        filepath = os.path.join(self.temp_dir, filename)
        
        doc = SimpleDocTemplate(filepath, pagesize=letter,
                               topMargin=1*inch, bottomMargin=1*inch,
                               leftMargin=1*inch, rightMargin=1*inch)
        
        story = []
        styles = getSampleStyleSheet()
        
        # Custom styles
        title_style = ParagraphStyle(
            'Title',
            parent=styles['Title'],
            fontSize=24,
            alignment=TA_CENTER,
            spaceAfter=30
        )
        
        heading1_style = ParagraphStyle(
            'Heading1',
            parent=styles['Heading1'],
            fontSize=18,
            spaceAfter=20
        )
        
        body_style = ParagraphStyle(
            'Body',
            parent=styles['BodyText'],
            fontSize=12,
            alignment=TA_JUSTIFY,
            spaceAfter=12,
            leading=24
        )
        
        # Title Page
        story.append(Paragraph(toc_data['title'], title_style))
        story.append(Spacer(1, 2*inch))
        
        author_style = ParagraphStyle('Author', parent=styles['Normal'], fontSize=14, alignment=TA_CENTER)
        story.append(Paragraph("Student Name<br/>Course Name<br/>Instructor Name", author_style))
        story.append(Spacer(1, 1*inch))
        story.append(Paragraph(datetime.now().strftime('%B %d, %Y'), author_style))
        story.append(PageBreak())
        
        # Table of Contents
        story.append(Paragraph("Table of Contents", heading1_style))
        
        toc_data_list = []
        page_num = 3
        for section in toc_data['sections']:
            if section['type'] == 'chapter':
                title = f"Chapter {section['number']}: {section['title']}"
            else:
                title = section['title']
            toc_data_list.append([title, str(page_num)])
            page_num += section['pages']
        
        toc_table = Table(toc_data_list, colWidths=[5*inch, 1*inch])
        toc_table.setStyle(TableStyle([
            ('FONT', (0, 0), (-1, -1), 'Times-Roman', 12),
            ('ALIGN', (1, 0), (1, -1), 'RIGHT'),
            ('VALIGN', (0, 0), (-1, -1), 'TOP'),
        ]))
        story.append(toc_table)
        story.append(PageBreak())
        
        # Content sections
        for section in toc_data['sections']:
            # Heading
            if section['type'] == 'chapter':
                heading_text = f"Chapter {section['number']}: {section['title']}"
            else:
                heading_text = section['title']
            
            story.append(Paragraph(heading_text, heading1_style))
            
            # Content
            if section['type'] != 'references':
                content = self.generate_content(section, topic)
                paragraphs = content.split('\n\n')
                
                for para_text in paragraphs:
                    if para_text.strip():
                        story.append(Paragraph(para_text.strip(), body_style))
            else:
                # References
                references = self.generate_references(topic)
                for ref in references:
                    ref_style = ParagraphStyle('Ref', parent=body_style, leftIndent=0.5*inch, firstLineIndent=-0.5*inch)
                    story.append(Paragraph(ref, ref_style))
            
            if section != toc_data['sections'][-1]:
                story.append(PageBreak())
        
        doc.build(story)
        return filepath


def generate_assignment(topic: str, page_count: int, format: str = 'docx') -> Dict[str, Any]:
    """
    Main function to generate assignment
    
    Args:
        topic: Assignment topic
        page_count: Number of pages (5-30)
        format: Output format (docx or pdf)
    
    Returns:
        {
            'success': bool,
            'toc': Dict,
            'file_path': str,
            'message': str
        }
    """
    
    try:
        # Validate
        if not topic or len(topic.strip()) < 5:
            return {
                'success': False,
                'message': 'Topic must be at least 5 characters long'
            }
        
        if not isinstance(page_count, int) or page_count < 5 or page_count > 30:
            return {
                'success': False,
                'message': 'Page count must be between 5 and 30'
            }
        
        if format not in ['docx', 'pdf']:
            return {
                'success': False,
                'message': 'Format must be docx or pdf'
            }
        
        # Generate
        generator = AssignmentGenerator()
        
        # Generate TOC
        toc_data = generator.generate_toc(topic, page_count)
        
        # Generate document
        if format == 'docx':
            filepath = generator.generate_docx(toc_data, topic)
        else:
            filepath = generator.generate_pdf(toc_data, topic)
        
        return {
            'success': True,
            'toc': toc_data,
            'file_path': filepath,
            'message': f'Assignment generated successfully as {format.upper()}'
        }
        
    except Exception as e:
        return {
            'success': False,
            'message': f'Error generating assignment: {str(e)}'
        }


if __name__ == '__main__':
    # Test
    result = generate_assignment(
        topic="The Impact of Artificial Intelligence on Modern Education",
        page_count=10,
        format='docx'
    )
    print(json.dumps(result, indent=2, default=str))
