"""
AI Assignment/Research Paper Generator
Generates academic documents with proper structure and APA references
"""

from docx import Document
from docx.shared import Inches, Pt
from docx.enum.text import WD_ALIGN_PARAGRAPH
from reportlab.lib.pagesizes import letter
from reportlab.platypus import SimpleDocTemplate, Paragraph, Spacer, PageBreak
from reportlab.lib.styles import getSampleStyleSheet, ParagraphStyle
from reportlab.lib.units import inch
import os
from datetime import datetime

def generate_toc(topic, page_count):
    """Generate Table of Contents based on topic and page count"""
    # Calculate chapters based on page count
    intro_pages = 1
    conclusion_pages = 1
    references_pages = 1
    content_pages = page_count - intro_pages - conclusion_pages - references_pages
    
    # Divide content into chapters
    if page_count <= 10:
        chapters = [
            "Introduction",
            f"Overview of {topic}",
            f"Key Concepts in {topic}",
            f"Analysis and Discussion",
            "Conclusion"
        ]
    elif page_count <= 20:
        chapters = [
            "Introduction",
            f"Background and Context of {topic}",
            f"Literature Review on {topic}",
            f"Theoretical Framework",
            f"Methodology and Approach",
            f"Key Findings and Analysis",
            f"Discussion and Implications",
            f"Challenges and Limitations",
            "Conclusion and Recommendations"
        ]
    else:
        chapters = [
            "Introduction",
            f"Historical Background of {topic}",
            f"Literature Review",
            f"Theoretical Framework and Concepts",
            f"Research Methodology",
            f"Data Analysis Part I",
            f"Data Analysis Part II",
            f"Results and Findings",
            f"Discussion and Interpretation",
            f"Implications for Practice",
            f"Limitations of the Study",
            f"Future Research Directions",
            "Conclusion and Recommendations"
        ]
    
    # Add References
    chapters.append("References")
    
    return chapters

def generate_chapter_content(chapter_title, topic, word_count):
    """Generate content for a chapter"""
    # Introduction content
    if "Introduction" in chapter_title:
        content = f"This research paper examines the multifaceted aspects of {topic} and its significance in contemporary contexts. "
        content += f"The purpose of this study is to provide a comprehensive analysis of {topic}, exploring its theoretical foundations, practical applications, and implications for future research and practice. "
        content += f"Through a systematic examination of relevant literature and empirical evidence, this paper aims to contribute to a deeper understanding of {topic} and its role in shaping current trends and developments. "
        content += f"\n\nThe scope of this research encompasses various dimensions of {topic}, including its historical evolution, key theoretical perspectives, and contemporary applications. "
        content += f"By synthesizing existing knowledge and identifying gaps in current understanding, this study seeks to advance scholarly discourse on {topic} and provide actionable insights for practitioners and policymakers. "
        content += f"The following chapters present a structured analysis of {topic}, beginning with foundational concepts and progressing to advanced applications and future directions."
    
    # Conclusion content
    elif "Conclusion" in chapter_title:
        content = f"This research paper has provided a comprehensive examination of {topic}, exploring its theoretical underpinnings, practical applications, and implications for various stakeholders. "
        content += f"The findings presented throughout this paper demonstrate the significance of {topic} in addressing contemporary challenges and opportunities. "
        content += f"Through systematic analysis and synthesis of relevant literature, this study has contributed to a deeper understanding of the complexities and nuances associated with {topic}. "
        content += f"\n\nThe key insights derived from this research highlight the importance of continued investigation into {topic} and its evolving role in shaping future developments. "
        content += f"Recommendations for practice include the implementation of evidence-based strategies informed by the findings of this study. "
        content += f"Future research should address the limitations identified in this paper and explore emerging dimensions of {topic} to advance both theoretical understanding and practical applications."
    
    # Regular chapter content
    else:
        content = f"This chapter examines {chapter_title.lower()} within the context of {topic}. "
        content += f"Understanding {chapter_title.lower()} is essential for developing a comprehensive perspective on {topic} and its various dimensions. "
        content += f"Through careful analysis and synthesis of relevant literature, this chapter presents key concepts, theories, and empirical evidence related to {chapter_title.lower()}. "
        content += f"\n\nThe theoretical foundations of {chapter_title.lower()} provide important insights into the mechanisms and processes underlying {topic}. "
        content += f"Research has demonstrated that {chapter_title.lower()} plays a crucial role in shaping outcomes and influencing various stakeholders involved in {topic}. "
        content += f"Contemporary perspectives on {chapter_title.lower()} emphasize the importance of integrating multiple approaches and methodologies to achieve a holistic understanding. "
        content += f"\n\nEmpirical studies have consistently shown that {chapter_title.lower()} is associated with significant implications for both theory and practice in the field of {topic}. "
        content += f"These findings suggest that practitioners and policymakers should consider {chapter_title.lower()} when designing interventions and strategies related to {topic}. "
        content += f"Furthermore, the evidence indicates that future research should continue to explore the nuances and complexities of {chapter_title.lower()} to advance our collective understanding of {topic}."
    
    # Ensure word count
    current_words = len(content.split())
    if current_words < word_count:
        # Add more content
        additional = f"\n\nFurther analysis reveals that {topic} encompasses multiple dimensions that warrant careful consideration. "
        additional += f"The interconnections between various aspects of {topic} highlight the complexity of the subject matter and the need for comprehensive approaches. "
        additional += f"Scholars have noted that {topic} continues to evolve in response to changing social, economic, and technological contexts. "
        additional += f"This dynamic nature necessitates ongoing research and adaptation of strategies to address emerging challenges and opportunities. "
        additional += f"The evidence presented in this section underscores the importance of maintaining a nuanced and contextually sensitive approach to understanding {topic}."
        
        content += additional
    
    # Trim if too long
    words = content.split()
    if len(words) > word_count:
        content = ' '.join(words[:word_count])
    
    return content

def generate_apa_references(topic, count=12):
    """Generate APA-style references"""
    references = []
    keywords = topic.split()
    main_keyword = keywords[0] if keywords else "Research"
    
    # Generate varied references
    authors = [
        ("Smith, J.", "Johnson, M."),
        ("Brown, L.", "Davis, K."),
        ("Wilson, R.", "Taylor, S."),
        ("Anderson, P.", "Thomas, C."),
        ("Martinez, A.", "Garcia, E."),
        ("Robinson, D.", "Lee, H.")
    ]
    
    journals = [
        f"Journal of {main_keyword} Studies",
        f"International Review of {main_keyword}",
        f"{main_keyword} Research Quarterly",
        f"Contemporary {main_keyword} Journal",
        f"Advances in {main_keyword} Theory",
        f"Applied {main_keyword} Science"
    ]
    
    for i in range(count):
        author_pair = authors[i % len(authors)]
        journal = journals[i % len(journals)]
        year = 2015 + (i % 9)
        vol = 10 + (i % 15)
        issue = 1 + (i % 4)
        pages_start = 100 + (i * 15)
        pages_end = pages_start + 20
        
        ref = f"{author_pair[0]}, & {author_pair[1]} ({year}). "
        ref += f"Investigating {topic.lower()}: A comprehensive analysis. "
        ref += f"{journal}, {vol}({issue}), {pages_start}-{pages_end}."
        
        references.append(ref)
    
    return references

def create_docx_assignment(topic, page_count, toc):
    """Generate Word document assignment"""
    doc = Document()
    
    # Set margins
    sections = doc.sections
    for section in sections:
        section.top_margin = Inches(1)
        section.bottom_margin = Inches(1)
        section.left_margin = Inches(1)
        section.right_margin = Inches(1)
    
    # Title Page
    title = doc.add_heading(topic, 0)
    title.alignment = WD_ALIGN_PARAGRAPH.CENTER
    
    doc.add_paragraph()
    doc.add_paragraph()
    
    author_para = doc.add_paragraph("Research Paper")
    author_para.alignment = WD_ALIGN_PARAGRAPH.CENTER
    
    date_para = doc.add_paragraph(datetime.now().strftime("%B %Y"))
    date_para.alignment = WD_ALIGN_PARAGRAPH.CENTER
    
    doc.add_page_break()
    
    # Table of Contents
    doc.add_heading('Table of Contents', 1)
    for i, chapter in enumerate(toc, 1):
        doc.add_paragraph(f"{i}. {chapter}", style='List Number')
    
    doc.add_page_break()
    
    # Calculate words per chapter
    words_per_page = 350
    total_words = page_count * words_per_page
    intro_words = 600
    conclusion_words = 600
    references_words = 300
    
    content_chapters = [ch for ch in toc if ch not in ['Introduction', 'Conclusion', 'References']]
    words_per_chapter = (total_words - intro_words - conclusion_words) // len(content_chapters)
    
    # Generate chapters
    for chapter_title in toc:
        if chapter_title == "References":
            continue
        
        doc.add_heading(chapter_title, 1)
        
        if chapter_title == "Introduction":
            word_count = intro_words
        elif chapter_title == "Conclusion":
            word_count = conclusion_words
        else:
            word_count = words_per_chapter
        
        content = generate_chapter_content(chapter_title, topic, word_count)
        doc.add_paragraph(content)
        doc.add_page_break()
    
    # References
    doc.add_heading('References', 1)
    references = generate_apa_references(topic)
    for ref in references:
        p = doc.add_paragraph(ref)
        p.paragraph_format.left_indent = Inches(0.5)
        p.paragraph_format.first_line_indent = Inches(-0.5)
    
    # Save
    output_path = f"outputs/assignment_{topic[:30].replace(' ', '_')}.docx"
    os.makedirs('outputs', exist_ok=True)
    doc.save(output_path)
    
    return output_path

def create_pdf_assignment(topic, page_count, toc):
    """Generate PDF assignment"""
    output_path = f"outputs/assignment_{topic[:30].replace(' ', '_')}.pdf"
    
    doc = SimpleDocTemplate(output_path, pagesize=letter,
                           topMargin=inch, bottomMargin=inch,
                           leftMargin=inch, rightMargin=inch)
    
    styles = getSampleStyleSheet()
    story = []
    
    # Title Page
    title_style = ParagraphStyle(
        'CustomTitle',
        parent=styles['Heading1'],
        fontSize=24,
        textColor='#1a1a1a',
        spaceAfter=30,
        alignment=1
    )
    story.append(Paragraph(topic, title_style))
    story.append(Spacer(1, 2*inch))
    story.append(Paragraph("Research Paper", styles['Normal']))
    story.append(Spacer(1, 0.5*inch))
    story.append(Paragraph(datetime.now().strftime("%B %Y"), styles['Normal']))
    story.append(PageBreak())
    
    # TOC
    story.append(Paragraph("Table of Contents", styles['Heading1']))
    for i, chapter in enumerate(toc, 1):
        story.append(Paragraph(f"{i}. {chapter}", styles['Normal']))
    story.append(PageBreak())
    
    # Chapters
    words_per_page = 350
    total_words = page_count * words_per_page
    content_chapters = [ch for ch in toc if ch not in ['Introduction', 'Conclusion', 'References']]
    words_per_chapter = (total_words - 1200) // len(content_chapters)
    
    for chapter_title in toc:
        if chapter_title == "References":
            continue
        
        story.append(Paragraph(chapter_title, styles['Heading1']))
        story.append(Spacer(1, 0.2*inch))
        
        word_count = 600 if chapter_title in ['Introduction', 'Conclusion'] else words_per_chapter
        content = generate_chapter_content(chapter_title, topic, word_count)
        
        story.append(Paragraph(content, styles['Normal']))
        story.append(PageBreak())
    
    # References
    story.append(Paragraph('References', styles['Heading1']))
    references = generate_apa_references(topic)
    for ref in references:
        story.append(Paragraph(ref, styles['Normal']))
        story.append(Spacer(1, 0.1*inch))
    
    doc.build(story)
    return output_path

def generate_assignment(topic, page_count, format_type, toc=None):
    """
    Main assignment generation function
    
    Args:
        topic: Research topic
        page_count: Number of pages (5/10/20/30/40)
        format_type: 'docx' or 'pdf'
        toc: Optional custom Table of Contents
    
    Returns:
        Path to generated file
    """
    try:
        # Generate or use provided TOC
        if toc is None:
            toc = generate_toc(topic, page_count)
        
        # Generate document
        if format_type == 'docx':
            output_path = create_docx_assignment(topic, page_count, toc)
        elif format_type == 'pdf':
            output_path = create_pdf_assignment(topic, page_count, toc)
        else:
            raise ValueError(f"Unsupported format: {format_type}")
        
        return output_path
        
    except Exception as e:
        raise Exception(f"Assignment generation failed: {str(e)}")

# For testing
if __name__ == "__main__":
    pass
