"""
Complete Multi-Tool Backend API
Production-ready Flask server with all tools integrated
"""

from flask import Flask, request, jsonify, send_file
from flask_cors import CORS
import os
from werkzeug.utils import secure_filename
import traceback
from datetime import datetime

# Import all tool modules
from tools.presentation_generator_v2 import generate_presentation
from tools.assignment_generator_v2 import generate_assignment
from tools.logo_analyzer import analyze_logo
from tools.screenshot_ocr import extract_text_from_screenshot
from tools.resume_bullet_generator import generate_resume_bullets
from tools.pdf_cleaner import clean_pdf_page
from tools.caption_rewriter import rewrite_caption

# Initialize Flask app
app = Flask(__name__)
CORS(app)  # Enable CORS for all routes

# Configuration
UPLOAD_FOLDER = '/tmp/uploads'
MAX_FILE_SIZE = 50 * 1024 * 1024  # 50MB
ALLOWED_EXTENSIONS = {'png', 'jpg', 'jpeg', 'gif', 'bmp', 'pdf', 'docx', 'pptx'}

os.makedirs(UPLOAD_FOLDER, exist_ok=True)
app.config['UPLOAD_FOLDER'] = UPLOAD_FOLDER
app.config['MAX_CONTENT_LENGTH'] = MAX_FILE_SIZE


def allowed_file(filename):
    """Check if file extension is allowed"""
    return '.' in filename and filename.rsplit('.', 1)[1].lower() in ALLOWED_EXTENSIONS


def handle_error(error, status_code=500):
    """Standard error handler"""
    return jsonify({
        'success': False,
        'error': str(error),
        'timestamp': datetime.now().isoformat()
    }), status_code


# ==================== Health Check ====================

@app.route('/health', methods=['GET'])
def health_check():
    """Health check endpoint"""
    return jsonify({
        'status': 'healthy',
        'timestamp': datetime.now().isoformat(),
        'version': '2.0',
        'tools': [
            'presentation-generator',
            'assignment-generator',
            'logo-analyzer',
            'screenshot-ocr',
            'resume-bullets',
            'pdf-cleaner',
            'caption-rewriter'
        ]
    })


# ==================== AI Presentation Generator ====================

@app.route('/api/presentation/generate-outline', methods=['POST'])
def generate_presentation_outline():
    """Generate presentation outline"""
    try:
        data = request.get_json()
        
        topic = data.get('topic', '').strip()
        slide_count = int(data.get('slide_count', 5))
        
        if not topic:
            return handle_error('Topic is required', 400)
        
        if slide_count < 3 or slide_count > 20:
            return handle_error('Slide count must be between 3 and 20', 400)
        
        # Import here to generate outline only
        from tools.presentation_generator_v2 import PresentationGenerator
        generator = PresentationGenerator()
        outline = generator.generate_outline(topic, slide_count)
        
        return jsonify({
            'success': True,
            'outline': outline,
            'message': 'Outline generated successfully'
        })
        
    except Exception as e:
        traceback.print_exc()
        return handle_error(e)


@app.route('/api/presentation/generate', methods=['POST'])
def generate_presentation_file():
    """Generate complete presentation file"""
    try:
        data = request.get_json()
        
        topic = data.get('topic', '').strip()
        slide_count = int(data.get('slide_count', 5))
        format = data.get('format', 'pptx').lower()
        add_watermark = data.get('add_watermark', True)
        
        result = generate_presentation(topic, slide_count, format, add_watermark)
        
        if result['success']:
            # Return file info (frontend will fetch via download endpoint)
            return jsonify({
                'success': True,
                'file_id': os.path.basename(result['file_path']),
                'outline': result['outline'],
                'message': result['message']
            })
        else:
            return handle_error(result['message'], 400)
        
    except Exception as e:
        traceback.print_exc()
        return handle_error(e)


@app.route('/api/presentation/download/<file_id>', methods=['GET'])
def download_presentation(file_id):
    """Download generated presentation"""
    try:
        file_path = f'/tmp/presentations/{secure_filename(file_id)}'
        
        if not os.path.exists(file_path):
            return handle_error('File not found', 404)
        
        return send_file(
            file_path,
            as_attachment=True,
            download_name=file_id
        )
        
    except Exception as e:
        traceback.print_exc()
        return handle_error(e)


# ==================== AI Assignment Generator ====================

@app.route('/api/assignment/generate-toc', methods=['POST'])
def generate_assignment_toc():
    """Generate assignment table of contents"""
    try:
        data = request.get_json()
        
        topic = data.get('topic', '').strip()
        page_count = int(data.get('page_count', 10))
        
        if not topic:
            return handle_error('Topic is required', 400)
        
        if page_count < 5 or page_count > 30:
            return handle_error('Page count must be between 5 and 30', 400)
        
        from tools.assignment_generator_v2 import AssignmentGenerator
        generator = AssignmentGenerator()
        toc_data = generator.generate_toc(topic, page_count)
        
        return jsonify({
            'success': True,
            'toc': toc_data,
            'message': 'Table of contents generated successfully'
        })
        
    except Exception as e:
        traceback.print_exc()
        return handle_error(e)


@app.route('/api/assignment/generate', methods=['POST'])
def generate_assignment_file():
    """Generate complete assignment file"""
    try:
        data = request.get_json()
        
        topic = data.get('topic', '').strip()
        page_count = int(data.get('page_count', 10))
        format = data.get('format', 'docx').lower()
        
        result = generate_assignment(topic, page_count, format)
        
        if result['success']:
            return jsonify({
                'success': True,
                'file_id': os.path.basename(result['file_path']),
                'toc': result['toc'],
                'message': result['message']
            })
        else:
            return handle_error(result['message'], 400)
        
    except Exception as e:
        traceback.print_exc()
        return handle_error(e)


@app.route('/api/assignment/download/<file_id>', methods=['GET'])
def download_assignment(file_id):
    """Download generated assignment"""
    try:
        file_path = f'/tmp/assignments/{secure_filename(file_id)}'
        
        if not os.path.exists(file_path):
            return handle_error('File not found', 404)
        
        return send_file(
            file_path,
            as_attachment=True,
            download_name=file_id
        )
        
    except Exception as e:
        traceback.print_exc()
        return handle_error(e)


# ==================== Logo Analyzer ====================

@app.route('/api/logo/analyze', methods=['POST'])
def analyze_logo_endpoint():
    """Analyze logo for improvements"""
    try:
        if 'file' not in request.files:
            return handle_error('No file uploaded', 400)
        
        file = request.files['file']
        
        if file.filename == '':
            return handle_error('No file selected', 400)
        
        if not allowed_file(file.filename):
            return handle_error('Invalid file type', 400)
        
        # Save file
        filename = secure_filename(file.filename)
        filepath = os.path.join(app.config['UPLOAD_FOLDER'], filename)
        file.save(filepath)
        
        # Analyze
        result = analyze_logo(filepath)
        
        # Cleanup
        os.remove(filepath)
        
        return jsonify(result)
        
    except Exception as e:
        traceback.print_exc()
        return handle_error(e)


# ==================== Screenshot OCR ====================

@app.route('/api/screenshot/extract-text', methods=['POST'])
def extract_text_endpoint():
    """Extract and clean text from screenshot"""
    try:
        if 'file' not in request.files:
            return handle_error('No file uploaded', 400)
        
        file = request.files['file']
        
        if file.filename == '':
            return handle_error('No file selected', 400)
        
        # Save file
        filename = secure_filename(file.filename)
        filepath = os.path.join(app.config['UPLOAD_FOLDER'], filename)
        file.save(filepath)
        
        # Extract text
        result = extract_text_from_screenshot(filepath)
        
        # Cleanup
        os.remove(filepath)
        
        return jsonify(result)
        
    except Exception as e:
        traceback.print_exc()
        return handle_error(e)


# ==================== Resume Bullet Generator ====================

@app.route('/api/resume/generate-bullets', methods=['POST'])
def generate_bullets_endpoint():
    """Generate professional resume bullet points"""
    try:
        data = request.get_json()
        
        job_description = data.get('job_description', '').strip()
        experience = data.get('experience', '').strip()
        bullet_count = int(data.get('bullet_count', 5))
        
        if not job_description:
            return handle_error('Job description is required', 400)
        
        result = generate_resume_bullets(job_description, experience, bullet_count)
        
        return jsonify(result)
        
    except Exception as e:
        traceback.print_exc()
        return handle_error(e)


# ==================== PDF Page Cleaner ====================

@app.route('/api/pdf/clean-page', methods=['POST'])
def clean_pdf_endpoint():
    """Clean/extract specific page from PDF"""
    try:
        if 'file' not in request.files:
            return handle_error('No file uploaded', 400)
        
        file = request.files['file']
        page_number = int(request.form.get('page_number', 1))
        
        if file.filename == '':
            return handle_error('No file selected', 400)
        
        # Save file
        filename = secure_filename(file.filename)
        filepath = os.path.join(app.config['UPLOAD_FOLDER'], filename)
        file.save(filepath)
        
        # Clean
        result = clean_pdf_page(filepath, page_number)
        
        # Cleanup
        os.remove(filepath)
        
        if result['success']:
            return send_file(
                result['output_path'],
                as_attachment=True,
                download_name=f'cleaned_page_{page_number}.pdf'
            )
        else:
            return handle_error(result['message'], 400)
        
    except Exception as e:
        traceback.print_exc()
        return handle_error(e)


# ==================== Caption Rewriter ====================

@app.route('/api/caption/rewrite', methods=['POST'])
def rewrite_caption_endpoint():
    """Rewrite social media caption"""
    try:
        data = request.get_json()
        
        caption = data.get('caption', '').strip()
        platform = data.get('platform', 'instagram').lower()
        tone = data.get('tone', 'engaging').lower()
        
        if not caption:
            return handle_error('Caption is required', 400)
        
        result = rewrite_caption(caption, platform, tone)
        
        return jsonify(result)
        
    except Exception as e:
        traceback.print_exc()
        return handle_error(e)


# ==================== Error Handlers ====================

@app.errorhandler(404)
def not_found(error):
    return handle_error('Endpoint not found', 404)


@app.errorhandler(413)
def too_large(error):
    return handle_error('File too large (max 50MB)', 413)


@app.errorhandler(500)
def internal_error(error):
    return handle_error('Internal server error', 500)


# ==================== Main ====================

if __name__ == '__main__':
    print("=" * 60)
    print("Multi-Tool Backend API Server")
    print("=" * 60)
    print("\nAvailable endpoints:")
    print("  GET  /health - Health check")
    print("\nAI Tools:")
    print("  POST /api/presentation/generate-outline - Generate presentation outline")
    print("  POST /api/presentation/generate - Generate presentation file")
    print("  GET  /api/presentation/download/<file_id> - Download presentation")
    print("  POST /api/assignment/generate-toc - Generate assignment TOC")
    print("  POST /api/assignment/generate - Generate assignment file")
    print("  GET  /api/assignment/download/<file_id> - Download assignment")
    print("\nImage & Document Tools:")
    print("  POST /api/logo/analyze - Analyze logo")
    print("  POST /api/screenshot/extract-text - Extract text from screenshot")
    print("  POST /api/resume/generate-bullets - Generate resume bullets")
    print("  POST /api/pdf/clean-page - Clean PDF page")
    print("  POST /api/caption/rewrite - Rewrite social caption")
    print("\n" + "=" * 60)
    print("\nStarting server on http://0.0.0.0:5000")
    print("=" * 60 + "\n")
    
    app.run(host='0.0.0.0', port=5000, debug=True)
